<?php
/**
 * DASHBOARD PRINCIPAL - DOMAIN MANAGER
 * 
 * Panel administrativo completo
 * Fecha: 2025-11-12
 * Versión: 1.0 Stable
 */

// Incluir archivos de configuración
require_once '../config.php';
require_once '../auth.php';
requireAuth();

// Obtener variables de sesión
$userId = $_SESSION['user_id'] ?? 1;
$username = $_SESSION['username'] ?? 'admin';

/**
 * Función para formatear fechas (definida localmente)
 */
function formatDate($date, $format = 'd/m/Y') {
    if (empty($date) || $date === '0000-00-00') {
        return 'Sin fecha';
    }
    
    try {
        $datetime = new DateTime($date);
        return $datetime->format($format);
    } catch (Exception $e) {
        return $date;
    }
}

// Obtener estadísticas de dominios
try {
    require_once '../functions.php';
    $pdo = getDBConnection();
    
    // Total de dominios activos
    $stmt = $pdo->query("SELECT COUNT(*) FROM domains WHERE status = 1");
    $totalDomains = $stmt->fetchColumn();
    
    // Dominios próximos a vencer (30 días)
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM domains WHERE status = 1 AND expiry_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)");
    $stmt->execute();
    $expiringSoon = $stmt->fetchColumn();
    
    // Dominios vencidos
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM domains WHERE status = 1 AND expiry_date < CURDATE()");
    $stmt->execute();
    $expiredDomains = $stmt->fetchColumn();
    
    // Registradores únicos
    $stmt = $pdo->query("SELECT COUNT(DISTINCT registrar) FROM domains WHERE registrar IS NOT NULL AND registrar != ''");
    $totalRegistrars = $stmt->fetchColumn();
    
    // Próximos vencimientos
    $stmt = $pdo->query("
        SELECT 
            d.*,
            DATEDIFF(d.expiry_date, CURDATE()) as days_until_expiry
        FROM domains d 
        WHERE d.status = 1 
        AND d.expiry_date IS NOT NULL 
        AND d.expiry_date != '0000-00-00'
        ORDER BY d.expiry_date ASC 
        LIMIT 5
    ");
    $upcomingExpiries = $stmt->fetchAll();
    
    // Actividad reciente
    $stmt = $pdo->query("SELECT action, details, created_at FROM activity_log ORDER BY created_at DESC LIMIT 5");
    $recentActivity = $stmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Error loading dashboard: " . $e->getMessage());
    $totalDomains = $expiringSoon = $expiredDomains = $totalRegistrars = 0;
    $recentActivity = $upcomingExpiries = [];
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Domain Manager</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --bg-primary: #1a1a1a;
            --bg-secondary: #2d2d2d;
            --bg-tertiary: #404040;
            --text-primary: #ffffff;
            --text-secondary: #cccccc;
            --accent-blue: #007bff;
            --accent-green: #28a745;
            --accent-yellow: #ffc107;
            --accent-red: #dc3545;
        }
        
        body {
            background: var(--bg-primary);
            color: var(--text-primary);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .sidebar {
            background: var(--bg-secondary);
            min-height: 100vh;
            box-shadow: 2px 0 10px rgba(0, 0, 0, 0.3);
        }
        
        .sidebar .nav-link {
            color: var(--text-secondary);
            padding: 0.75rem 1rem;
            margin: 0.25rem 0;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        
        .sidebar .nav-link:hover,
        .sidebar .nav-link.active {
            background: var(--bg-tertiary);
            color: var(--text-primary);
        }
        
        .main-content {
            padding: 2rem;
        }
        
        .card {
            background: var(--bg-secondary);
            border: 1px solid var(--bg-tertiary);
            border-radius: 12px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .card-header {
            background: var(--bg-tertiary);
            border-bottom: 1px solid var(--bg-tertiary);
            color: var(--text-primary);
            font-weight: 600;
        }
        
        .stats-card {
            background: linear-gradient(135deg, var(--bg-secondary) 0%, var(--bg-tertiary) 100%);
            border: 1px solid var(--accent-blue);
            transition: transform 0.3s ease;
        }
        
        .stats-card:hover {
            transform: translateY(-2px);
        }
        
        .stats-number {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .stats-label {
            font-size: 0.9rem;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .navbar {
            background: var(--bg-secondary);
            border-bottom: 1px solid var(--bg-tertiary);
        }
        
        .badge {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            font-size: 0.8rem;
        }
        
        .badge.bg-success { background: var(--accent-green) !important; }
        .badge.bg-warning { background: var(--accent-yellow) !important; color: var(--bg-primary) !important; }
        .badge.bg-danger { background: var(--accent-red) !important; }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid var(--bg-tertiary);
            border-radius: 50%;
            border-top-color: var(--accent-blue);
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin { to { transform: rotate(360deg); } }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-3">
                    <h4 class="text-center mb-4">
                        <i class="fas fa-globe text-primary"></i>
                        Domain Manager
                    </h4>
                    
                    <nav class="nav flex-column">
                        <a class="nav-link active" href="index.php">
                            <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                        </a>
                        <hr class="my-3" style="border-color: var(--bg-tertiary);">
                        <a class="nav-link" href="../index.php?logout=1">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </nav>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 main-content">
                <!-- Header -->
                <div class="navbar navbar-expand-lg mb-4">
                    <div class="container-fluid">
                        <div class="navbar-nav ms-auto">
                            <div class="nav-item dropdown">
                                <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                                    <i class="fas fa-user-circle me-2"></i><?php echo htmlspecialchars($username); ?>
                                </a>
                                <ul class="dropdown-menu">
                                    <li><a class="dropdown-item" href="../index.php?logout=1">Cerrar Sesión</a></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Page Header -->
                <div class="page-header">
                    <h1><i class="fas fa-tachometer-alt me-2"></i>Dashboard</h1>
                    <p class="text-muted">Gestión de dominios - Sistema operativo</p>
                </div>
                
                <!-- Alerts -->
                <?php if ($expiredDomains > 0): ?>
                    <div class="alert alert-danger mb-4">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>¡Atención!</strong> Tienes <?php echo $expiredDomains; ?> dominio(s) vencido(s).
                    </div>
                <?php endif; ?>
                
                <?php if ($expiringSoon > 0): ?>
                    <div class="alert alert-warning mb-4">
                        <i class="fas fa-clock me-2"></i>
                        <strong>Recordatorio:</strong> <?php echo $expiringSoon; ?> dominio(s) vencerán en 30 días.
                    </div>
                <?php endif; ?>
                
                <!-- Stats Cards -->
                <div class="row mb-4">
                    <div class="col-md-3 mb-3">
                        <div class="card stats-card h-100">
                            <div class="card-body text-center">
                                <div class="stats-number text-primary"><?php echo number_format($totalDomains); ?></div>
                                <div class="stats-label">Total Dominios</div>
                                <i class="fas fa-globe fa-2x text-muted mt-3"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3 mb-3">
                        <div class="card stats-card h-100">
                            <div class="card-body text-center">
                                <div class="stats-number text-warning"><?php echo number_format($expiringSoon); ?></div>
                                <div class="stats-label">Próximos a Vencer</div>
                                <i class="fas fa-clock fa-2x text-muted mt-3"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3 mb-3">
                        <div class="card stats-card h-100">
                            <div class="card-body text-center">
                                <div class="stats-number text-danger"><?php echo number_format($expiredDomains); ?></div>
                                <div class="stats-label">Dominios Vencidos</div>
                                <i class="fas fa-exclamation-triangle fa-2x text-muted mt-3"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3 mb-3">
                        <div class="card stats-card h-100">
                            <div class="card-body text-center">
                                <div class="stats-number text-success"><?php echo number_format($totalRegistrars); ?></div>
                                <div class="stats-label">Registradores</div>
                                <i class="fas fa-building fa-2x text-muted mt-3"></i>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Next Expiries -->
                <div class="row">
                    <div class="col-md-6">
                        <div class="card h-100">
                            <div class="card-header">
                                <i class="fas fa-calendar-alt me-2"></i>Próximos Vencimientos
                            </div>
                            <div class="card-body">
                                <?php if (empty($upcomingExpiries)): ?>
                                    <p class="text-muted">No hay vencimientos próximos</p>
                                <?php else: ?>
                                    <?php foreach ($upcomingExpiries as $domain): ?>
                                        <div class="d-flex justify-content-between align-items-center mb-3">
                                            <div>
                                                <strong><?php echo htmlspecialchars($domain['name'] . '.' . $domain['extension']); ?></strong>
                                                <br>
                                                <small class="text-muted"><?php echo htmlspecialchars($domain['registrar'] ?? 'Sin registrar'); ?></small>
                                            </div>
                                            <div class="text-end">
                                                <?php if ($domain['days_until_expiry'] < 0): ?>
                                                    <span class="badge bg-danger">Vencido</span>
                                                <?php elseif ($domain['days_until_expiry'] <= 30): ?>
                                                    <span class="badge bg-warning"><?php echo $domain['days_until_expiry']; ?> días</span>
                                                <?php else: ?>
                                                    <span class="badge bg-success"><?php echo $domain['days_until_expiry']; ?> días</span>
                                                <?php endif; ?>
                                                <br>
                                                <small class="text-muted"><?php echo formatDate($domain['expiry_date']); ?></small>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card h-100">
                            <div class="card-header">
                                <i class="fas fa-history me-2"></i>Actividad Reciente
                            </div>
                            <div class="card-body">
                                <?php if (empty($recentActivity)): ?>
                                    <p class="text-muted">No hay actividad reciente</p>
                                <?php else: ?>
                                    <?php foreach ($recentActivity as $activity): ?>
                                        <div class="mb-2">
                                            <strong><?php echo htmlspecialchars($activity['action']); ?></strong>
                                            <br>
                                            <small class="text-muted"><?php echo formatDate($activity['created_at'], 'd/m/Y H:i'); ?></small>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>